mod common;
mod srp_analyser;

use common::*;
use srp_analyser::*;

use std::path::PathBuf;
use structopt::StructOpt;

use std::fs::File;
use std::io::prelude::*;


#[derive(Debug, StructOpt)]
#[structopt(name = "srp_analysis", about = "Preforms srp analysis.")]
struct Opt {
    /// Approximate preemptions
    #[structopt(short, long)]
    approx: bool,    
    
    /// Input file, example input if not present
    #[structopt(parse(from_os_str))]
    input: Option<PathBuf>,

    /// Output file, stdout if not present
    #[structopt(parse(from_os_str))]
    output: Option<PathBuf>,
}


fn main() {

    let opt = Opt::from_args();
    //println!("{:?}", opt);
    
    let tasks: Tasks = match opt.input {
        Some(file) => load_tasks(file),
        None => create_tasks(),
    };
    
    let (ip, tr) = pre_analysis(&tasks);
    let analysis = analyse(&tasks, &ip, &tr, opt.approx);

    match opt.output {
        Some(file) => {
            write_to_file(serde_json::to_string(&analysis).unwrap(), file);
        }, // TODO: Implement so that the analysis can be saved to txt.
        None => {
            println!("{}", format_analysis(&analysis, &tasks, &ip, &tr)); 
        },
    };
}


fn format_analysis(analysis: &Vec<AnalysedTask>, tasks: &Tasks, ip: &IdPrio, tr: &TaskResources) -> String {
    let overview = format_overview(tasks, ip, tr);

    let mut body = String::new();
    for at in analysis {
        body = format!("{}\n{}", body, format_analysed_task(at));
    }

    body = format!("{}\n", body);

    for t in tasks {
        body = format!("{}\n{}", body, format_task(t));
    }

    return format!("{}\n{}", overview, body);
}


fn format_overview(tasks: &Tasks, ip: &IdPrio, tr: &TaskResources) -> String {
    fn format_task_names(tasks: &Tasks) -> String {
        let mut task_names = "Tasks:".to_string();
        for t in tasks {
            task_names = format!("{} {}", task_names, t.id);
        } 
        return task_names;
    }
    fn format_id_prio(ip: &IdPrio) -> String {
//    let id_prio = format!("Id Priority: {:?}", ip);
        let mut id_prio = "Priority:".to_string();
        for (key, value) in ip {
            id_prio = format!("{}  P({}) = {}", id_prio, key, value);
        }
        return id_prio;
    }
    fn format_task_res(tr: &TaskResources) -> String {
//    let task_res = format!("Task Resources: {:?}", tr);
        let mut task_res = "Task Resources:".to_string();
        for (key, value) in tr {
            
            let mut values_st = "(".to_string();
            for v in value {
                values_st = format!("{} {}", values_st, v);
            }
            task_res = format!("{} {}:{} )", task_res, key, values_st);
        }
        return task_res;
    }
    let tot_util = format!("Total Util: {}", total_load_factor(tasks));
    let task_names = format_task_names(tasks);
    let id_prio = format_id_prio(ip);
    let task_res = format_task_res(tr);

    return format!("Overview\n\t{}\n\t{}\n\t{}\n\t{}", tot_util, task_names, id_prio, task_res); 
}
 

fn format_analysed_task(task: &AnalysedTask) -> String {
    let header = format!("Analysis of {}", task.task.id);
    let res_t = format!("Response Time: {}", task.response_time);
    let wcet = format!("wcet: {}", task.wcet);
    let block_t = format!("Block Time: {}", task.block_time);
    let pre_t = format!("Preemtion Time: {}", task.preemtion_time);
    return format!("{}\n\t{}\n\t{}\n\t{}\n\t{}", header, res_t, wcet, block_t, pre_t);
}

fn format_task(task: &Task) -> String {
    let id = format!("Task {}", task.id);
    let prio = format!("Priority: {}", task.prio);
    let dl = format!("Deadline: {}", task.deadline);
    let inter_a = format!("Inter Arrival: {}", task.inter_arrival);
    let trace = format_trace(&task.trace);
    return format!("{}\n\t{}\n\t{}\n\t{}\n\t{}", id, prio, dl, inter_a, trace);
}

fn format_trace(trace: &Trace) -> String {
    return format!("{:?}", trace);
}



fn write_to_file(contents: String, file: PathBuf) -> std::io::Result<()> {
    let mut file = File::create(file)?;
    file.write_all(contents.as_bytes())?;
    Ok(())
}

fn load_tasks(file: PathBuf) -> Tasks {
    let mut serialized = String::new();
    read_from_file(&mut serialized, file);
    return serde_json::from_str(&serialized).unwrap();
}

fn read_from_file(contents: &mut String, filename: PathBuf) -> std::io::Result<()> {
    let mut file = File::open(filename)?;
    file.read_to_string(contents)?;
    Ok(())
}


fn create_tasks() -> Tasks { let t1 = Task {
        id: "T1".to_string(),
        prio: 1,
        deadline: 100,
        inter_arrival: 100,
        trace: Trace {
            id: "T1".to_string(),
            start: 0,
            end: 10,
            inner: vec![],
        },
    };

    let t2 = Task {
        id: "T2".to_string(),
        prio: 2,
        deadline: 200,
        inter_arrival: 200,
        trace: Trace {
            id: "T2".to_string(),
            start: 0,
            end: 30,
            inner: vec![
                Trace {
                    id: "R1".to_string(),
                    start: 10,
                    end: 20,
                    inner: vec![Trace {
                        id: "R2".to_string(),
                        start: 12,
                        end: 16,
                        inner: vec![],
                    }],
                },
                Trace {
                    id: "R1".to_string(),
                    start: 22,
                    end: 28,
                    inner: vec![],
                },
            ],
        },
    };

    let t3 = Task {
        id: "T3".to_string(),
        prio: 3,
        deadline: 50,
        inter_arrival: 50,
        trace: Trace {
            id: "T3".to_string(),
            start: 0,
            end: 30,
            inner: vec![Trace {
                id: "R2".to_string(),
                start: 10,
                end: 20,
                inner: vec![],
            }],
        },
    };
    
    // builds a vector of tasks t1, t2, t3
    let tasks: Tasks = vec![t1, t2, t3];
    return tasks;
}

